--[[
 * Copyright (c) 2012-2016 Those Awesome Guys (TAG)
 *
 * You may NOT use this software for any commercial
 * purpose, nor can you alter and redistribute it as your own.
 * Any credit must be attributed to Those Awesome Guys (TAG)
 *
 * You are allowed and even encouraged to mod(ify) the game
 * but all alterations must be specificed as a mod of the game
 * Move or Die, and not claim that Move or Die itself is a game
 * of your creation.
 *
 *  This notice may not be removed or altered from any source distribution.
 --]]

--Look up all code and assets in User/ directory, if not found, get the built-in one from the game as normal


require("Engine/GlobalVariables"); ---initalize all the global variables to be used anywhere in the code.

local events = require("Engine/Events") --the event system for detecting key presses, mouse presses and such
local gamesetup = require("Engine/GameSetup")
local accum = 0
local TimeStep = (1/60)
local shouldDraw = false
local customE = require("Engine/CustomError")
local networking = require("Src/Server/Networking")
local steam = require("Engine/Steam")
local popup = require("Src/Popup")
local analytics = require("Engine/Analytics")
local modAnalytics = require("Src/Server/MoDAnalytics")
local socket = require("socket")
local PerfCounters = require("Engine/PerformanceCounters")
local Effects = require("Engine/Effects")
local DiscordRPC = require("Engine/DiscordRPC")

_defaultErrhand = love.errhand;

---cap variables
local next_time = 0;

function love.load(args)

	_Dir = args[1] .. "/"

	local connectIndex = nil
	local connectLobbyIndex = nil
	local LANcommand = nil
	local stressTest = nil
	local convention = nil

	for i,arg in pairs(args) do
		if arg == "+connect" then
			connectIndex = i
			break
		elseif arg == "+connect_lobby_b64" or arg == "+connect_lobby" then
			connectLobbyIndex = i
			break
		elseif arg == "+host_lan_game" then
			LANcommand = "host"
		elseif arg == "+join_lan_game" then
			LANcommand = "join"
		elseif arg == "+editor" then
			_PARTYEDITOR = true
		elseif arg == "convention" then
			convention = "Convention"
		elseif arg == "arcade" then
			convention = "Arcade"
		elseif arg == "+stress" then
			stressTest = true
		end
	end

	if not _PARTYEDITOR then _PARTY = true end

	steam:init();
	DiscordRPC:init();
	gamesetup:init();

	if _PARTYEDITOR then
		DiscordRPC:updatePresence({
			largeImageKey = "misc_editor",
			state = "Creating Dank Mods",
			details = "In the Level Editor",
			startTimestamp = DiscordRPC.presence and DiscordRPC.presence.startTimestamp or DiscordRPC:getTime(0)
		})
	end

	if convention then _CONVENTION = convention end

	if connectIndex ~= nil or connectLobbyIndex ~= nil or LANcommand or _PARTYEDITOR then
		_StartMode = "Loading"
	elseif stressTest then
		_StartMode = "StressTest"
	end

	gamesetup:loadMode(_StartMode);
	next_time = love.timer.getTime();

	local menu = gamesetup:getCurrentMode().getMenu and gamesetup:getCurrentMode().getMenu()

	if menu then
		if connectIndex ~= nil and connectIndex < #args then
			menu.hostIP = args[connectIndex+1]
			menu.autoJoinLobby = true
		elseif connectLobbyIndex ~= nil and connectLobbyIndex < #args then
			menu.hostLobby = args[connectLobbyIndex+1]
			menu.autoJoinLobby = true
		else
			menu.LANcommand = LANcommand
		end
	end

	collectgarbage ("stop");

end

function love.draw()
	PerfCounters.start("render")
	gamesetup:Render();
	PerfCounters.stop("render")
end

local ms = 0;

 function love.errhand(msg)
-- --     ---A stackoverflow will cause the game to close without any error
	customE:handler(msg)
 end

 function love.threaderror(thread, errorstr)
  print("Thread error!\n"..errorstr)
  -- thread:getError() will return the same error string now.
end

local num = 0;
local prevTime = 0;
local realDelta = 1;

function love.resetTime()
	accum = 0
	prevTime = love.timer.getTime()
end

function love.update(dt)

	PerfCounters.start("update")

	--local start = os.clock()
	accum = accum + dt
	if accum > 1.0 then accum = TimeStep end -- we're more than a second behind, just do the step already
	local loops = 0
	shouldDraw = false;
	while(accum >= TimeStep)do ---makes it wait in-between frames if fps is above 60, and makes it skip frames if fps is below 60
		if Effects.clearJoeData then Effects:clearJoeData() end -- only clear this when we know the game might be updating it
		gamesetup:Update(TimeStep,1/dt);
		accum = (accum -TimeStep)
		loops = loops + 1
		shouldDraw = true;
		-- if(loops > 2) then accum = 0 end--adds a limit of 3 frames skipped, after that, slow down game
		local delta = love.timer.getTime() - prevTime;
		realDelta = delta;
		prevTime = love.timer.getTime()
	end

	-- Update without fixed timestep
	DiscordRPC:update(dt)
	if Effects.update then Effects:update(dt) end

	local done = collectgarbage("step",5)
	if(done)then
		num = num + 1;
		--print("Garbage Collected " .. num);
	end

	PerfCounters.stop("update")

--	local stop = os.clock()

--	local time = stop - start

--	if time > 0.5 then
--		print( "frame took "..tostring(stop - start))
--	end


end

function love.resize()
	gamesetup:ResizeEvent()
end

function love.keypressed(key, scancode)
	events:KeyDown(key, scancode);
end

function love.textinput(t)
	events:TextInput(t)
end

function love.keyreleased(key, scancode)
	events:KeyUp(key, scancode);
end

function love.touchpressed(id, x, y, dx, dy, pressure)
	-- print("touchpressed", tostring(id), x, y, dx, dy, pressure)
end

function love.touchreleased(id, x, y, dx, dy, pressure)
	-- print("touchreleased", tostring(id), x, y, dx, dy, pressure)
end

function love.touchmoved(id, x, y, dx, dy, pressure)
	-- print("touchmoved", tostring(id), x, y, dx, dy, pressure)
end

function love.wheelmoved(dx,dy)
	events:MouseScroll(dx,dy)
end

function love.mousepressed(x,y,button)
	events:MouseDown(x,y,button);
end

function love.mousereleased(x,y,button)
	events:MouseUp(x,y,button);
end

function love.joystickadded(joystick)
	events:JoystickAdded(joystick)
end

function love.joystickremoved(joystick)
	events:JoystickRemoved(joystick)
end

function love.focus(bool)
	events:Focus(bool)
end

function startWatchdogThread()
	if _PARTYEDITOR then return end
	if _watchdogThread then return end
	_watchdogThread = love.thread.newThread("Engine/WatchdogThread.lua")
	_watchdogThread:start()
end

love.quit = function()
	startWatchdogThread()
	networking:Disconnect()
	steam:update(0)
	steam:shutdown()
	DiscordRPC:shutdown()
	if Effects.shutdown then Effects:shutdown() end
	love.quit = nil
	love.event.quit()
end

function love.run()
	if love.math then
		love.math.setRandomSeed(socket.gettime() * 1e6)
	end

	math.randomseed(socket.gettime() * 1e6)
	math.random() math.random()

	if love.event then
		love.event.pump()
	end

	if love.load then love.load(arg) end

	-- We don't want the first frame's dt to include time taken by love.load.
	if love.timer then love.timer.step() end

	local dt = 0

	-- Main loop time.
	while true do
		-- Process events.
		if love.event then
			love.event.pump()
			for e,a,b,c,d in love.event.poll() do
				if e == "wrapup" then
					_CRASHED = false
					_Parser:write()

					if not _PARTYEDITOR then
						startWatchdogThread()

						-- close the window to make the user THINK he closed immediately
						love.window.close()

						-- audio should go down too
						if love.audio then
							love.audio.stop()
						end

						require("Src/Server/TwitchManager"):cleanup()

						networking:Disconnect()
						modAnalytics:removeHost()

						-- but secretly keep the app open for a short while longer to send analytics. We're not evil, this allows us to improve the game!
						local modAnalyticsFinished = true
						local isPartyMode = gamesetup and gamesetup:getCurrentMode() and gamesetup:getCurrentMode().partyStarted
		  				if isPartyMode then
		  					local partyMode = gamesetup:getCurrentMode()
		  					local abandon, rounds = partyMode:checkAbandon()

							if abandon then
			  					local success = function()
		  							modAnalyticsFinished = true
		  							print("logAbandon succeeded")
		  						end
			  					local fail = function()
		  							print("logAbandon failed, trying again")
		  							modAnalytics:logAbandon(abandon, rounds, success, function()
	  									print("logAbandon failed")
	  									modAnalyticsFinished = true
		  							end)
		  						end
			  					print("logging abandon on quit")
			  					modAnalyticsFinished = false
			  					modAnalytics:logAbandon(abandon, rounds, success, fail)
			  				end
		  				end

		  				analytics:sendSessionAnalytics()

						local menu = gamesetup and gamesetup:getCurrentMode() and gamesetup:getCurrentMode().getMenu and gamesetup:getCurrentMode().getMenu()
						if menu and menu.timeInMenu and menu.timeInMenu > 0 then
							analytics:logBlocking({"Time","Time_In_Menu","-",menu.timeInMenu})
							menu.timeInMenu = nil
						end

						local timeout = 0
						while (not modAnalyticsFinished or modAnalytics.hostParams) and timeout < 10 do
							love.timer.sleep(.1)
							timeout = timeout + .1
							modAnalytics:update()
						end
						if timeout >= 10 then
							print('logging abandon timed out')
						end
					end

					-- and THEN do the real quit
					if not love.quit or not love.quit() then
						return
					end
				end

				if e == "quit" then
					if not love.quit or not love.quit() then
						startWatchdogThread()
						if love.audio then
							love.audio.stop()
						end
						return
					end
				end
				if not _PARTYEDITOR or e ~= "wrapup" then love.handlers[e](a,b,c,d) end
			end
		end

		-- Update dt, as we'll be passing it to update
		if love.timer then
			love.timer.step()
			dt = love.timer.getDelta()
		end

		-- Call update and draw
		if love.update then love.update(dt) end -- will pass 0 if love.timer is disabled

		if love.window and love.graphics and love.window.isCreated() and shouldDraw then
			-- love.graphics.clear()
			love.graphics.origin()
			if love.draw then love.draw() end
			love.graphics.present()
			shouldDraw = false;
		end

		local cap = (_FPSCap == 0) and 0 or (1/_FPScap);
		next_time = next_time + cap;

		local cur_time = love.timer.getTime()
		if next_time <= cur_time then
			next_time = cur_time
		else
			love.timer.sleep(next_time - cur_time)
		end

	end

end
